<?php
/**
 * Manage translations in Listify.
 *
 * To avoid making users translate Listify and the WP Job Manager plugin
 * (which contains a lot of irrelevant strings) we can add each of the
 * plugin's strings to Listify's manifest.
 *
 * @since 1.0.0
 *
 * @package Listify
 * @category i18n
 * @author Astoundify
 */

// @codingStandardsIgnoreFile
class Listify_Strings {

	/**
	 * Registered strings.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	public $strings;

	/**
	 * Singular and plural labels.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	public $labels;

	/**
	 * Whitelist of text domains.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	public $domains;

	/**
	 * Hook in to WordPress.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		if ( ! apply_filters( 'listify_use_custom_strings', true ) ) {
			return;
		}

		add_action( 'after_setup_theme', array( $this, 'setup' ) );
	}

	/**
	 * Register whitelisted domains, strings, and labels.
	 *
	 * @since 1.0.0
	 */
	public function setup() {
		$this->labels = array(
			'singular' => get_theme_mod( 'label-singular', 'Company' ),
			'plural'   => get_theme_mod( 'label-plural', 'Companies' ),
		);

		$this->strings = $this->get_strings();

		$this->domains = apply_filters(
			'listify_gettext_domains', array(
				'wp-job-manager',
				'wp-job-manager-alerts',
				'wp-job-manager-locations',
				'wp-job-manager-wc-paid-listings',
				'wp-job-manager-simple-paid-listings',
				'wp-job-manager-extended-location',
			)
		);

		$this->translations = get_translations_for_domain( 'listify' );

		add_filter( 'gettext', array( $this, 'gettext' ), 0, 3 );
		add_filter( 'gettext_with_context', array( $this, 'gettext_with_context' ), 0, 4 );
		add_filter( 'ngettext', array( $this, 'ngettext' ), 0, 5 );
	}

	/**
	 * Get a label depending on the chosen form (plural or singular).
	 *
	 * @since 1.0.0
	 *
	 * @param string $form Singular or plural.
	 * @param bool   $slug Create a slug from the noun.
	 * @return string
	 */
	public function label( $form, $slug = false ) {
		$label = $this->labels[ $form ];

		if ( '' === $label && 'plural' === $form ) {
			$label = 'Listings';
		} elseif ( '' === $label && 'singular' === $form ) {
			$label = 'Listing';
		}

		if ( ! $slug ) {
			return $label;
		}

		return sanitize_title( $label );
	}

	/**
	 * Translate a string.
	 *
	 * @since 1.0.0
	 *
	 * @param string $string The string to translate.
	 * @return string
	 */
	private function translate_string( $string ) {
		$value = $string;
		$array = is_array( $value );

		$to_translate = $array ? $value[0] : $value;

		$translated = $this->translations->translate( $to_translate );

		if ( ! $translated ) {
			return $string;
		}

		if ( $array ) {
			$translated = vsprintf( $translated, $value[1] );
		}

		return $translated;
	}

	/**
	 * Translate a plural string.
	 *
	 * @since 1.0.0
	 *
	 * @param string $single Single form.
	 * @param string $plural Plural form.
	 * @param int    $number The number to check.
	 * @return string
	 */
	private function translate_plural( $single, $plural, $number ) {
		$translation = $this->translations->translate_plural( $single, $plural, $number );

		return $translation;
	}

	/**
	 * Filter standard gettext calls.
	 *
	 * @since 1.0.0
	 *
	 * @param string $translated The translated string loaded from a translation file.
	 * @param string $original The original string from the .pot file.
	 * @param string $domain The text domain.
	 * @return string
	 */
	public function gettext( $translated, $original, $domain ) {
		if ( ! in_array( $domain, $this->domains, true ) ) {
			return $translated;
		}

		if ( isset( $this->strings[ $domain ][ $original ] ) ) {
			return $this->translate_string( $this->strings[ $domain ][ $original ] );
		} else {
			return $translated;
		}
	}

	/**
	 * Filter gettext calls with context.
	 *
	 * @since 1.0.0
	 *
	 * @param string $translated The translated string loaded from a translation file.
	 * @param string $original The original string from the .pot file.
	 * @param string $context The context of the original string.
	 * @param string $domain The text domain.
	 * @return string
	 */
	public function gettext_with_context( $translated, $original, $context, $domain ) {
		if ( ! in_array( $domain, $this->domains, true ) ) {
			return $translated;
		}

		if ( isset( $this->strings[ $domain ][ $original ] ) ) {
			return $this->translate_string( $this->strings[ $domain ][ $original ] );
		} else {
			return $translated;
		}
	}

	/**
	 * Filter plural gettext calls.
	 *
	 * @since 1.0.0
	 *
	 * @param string $original The original string from the .pot file.
	 * @param string $single The singular version of the string.
	 * @param string $plural The plural version of the string.
	 * @param int    $number The number to check against.
	 * @param string $domain The text domain.
	 * @return string
	 */
	public function ngettext( $original, $single, $plural, $number, $domain ) {
		if ( ! in_array( $domain, $this->domains, true ) ) {
			return $original;
		}

		if ( isset( $this->strings[ $domain ][ $single ] ) ) {
			$base   = $this->strings[ $domain ][ $single ];
			$single = $base[0];
			$plural = $base[1];

			return $this->translate_plural( $single, $plural, $number );
		} else {
			return $original;
		}
	}

	/**
	 * Create a manifest of strings.
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	private function get_strings() {
		$strings = array(
			'wp-job-manager'                      => array(
				'Job'                                     => $this->label( 'singular' ),
				'Jobs'                                    => $this->label( 'plural' ),

				// Translators: %s the singular "listing" label used in permalinks". only transalte this if you are using Polylang and need to force the slugs to remain in a single language.
				'job'                                     => utf8_uri_encode( sprintf( _x( '%s', 'the singular "listing" label used in permalinks". only transalte this if you are using Polylang and need to force the slugs to remain in a single language.', 'listify' ), $this->label( 'singular', true ) ) ),

				// Translators: %s the plural "listing" label used in permalinks". only transalte this if you are using Polylang and need to force the slugs to remain in a single language.
				'jobs'                                    => utf8_uri_encode( sprintf( _x( '%s', 'the plural "listing" label used in permalinks". only transalte this if you are using Polylang and need to force the slugs to remain in a single language.', 'listify' ), $this->label( 'plural', true ) ) ),

				'Job Listings'                            => $this->label( 'plural' ),

				'Job category'                            => array(
					__( '%s Category', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Job categories'                          => array(
					__( '%s Categories', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Job Categories'                          => array(
					__( '%s Categories', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'job-category'                            => array(
					__( '%s-category', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),

				'Job type'                                => array(
					__( '%s Type', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Job types'                               => array(
					__( '%s Types', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Job Types'                               => array(
					__( '%s Types', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'job-type'                                => array(
					__( '%s-type', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),

				'Jobs will be shown if within ANY selected category' => array(
					__( '%s will be shown if within ANY selected category', 'listify' ),
					array( $this->label( 'plural' ) ),
				),
				'Jobs will be shown if within ALL selected categories' => array(
					__( '%s will be shown if within ALL selected categories', 'listify' ),
					array( $this->label( 'plural' ) ),
				),

				'Application email'                       => __( 'Contact Email', 'listify' ),
				'Application email/URL'                   => __( 'Contact Email/URL', 'listify' ),
				'Application URL'                         => __( 'Contact URL', 'listify' ),
				'Application Email or URL'                => __( 'Contact email/URL', 'listify' ),
				'Position filled?'                        => __( 'Listing filled?', 'listify' ),
				'A video about your company'              => __( 'A video about your listing', 'listify' ),

				'Job Submission'                          => array(
					__( '%s Submission', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Submit Job Form Page'                    => array(
					__( 'Submit %s Form Page', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Job Dashboard Page'                      => array(
					__( '%s Dashboard Page', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Job Listings Page'                       => array(
					__( '%s Page', 'listify' ),
					array( $this->label( 'plural' ) ),
				),

				'Add a job via the back-end'              => array(
					__( 'Add a %s via the back-end', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),
				'Add a job via the front-end'             => array(
					__( 'Add a %s via the front-end', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),
				'Find out more about the front-end job submission form' => array(
					__( 'Find out more about the front-end %s submission form', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),
				'View submitted job listings'             => array(
					__( 'View submitted %s listings', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Add the [jobs] shortcode to a page to list jobs' => array(
					__( 'Add the [jobs] shortcode to a page to list %s', 'listify' ),
					array( $this->label( 'plural', true ) ),
				),
				'View the job dashboard'                  => array(
					__( 'View the %s dashboard', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),
				'Find out more about the front-end job dashboard' => array(
					__( 'Find out more about the front-end %s dashboard', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),

				'Company name'                            => __( 'Company name', 'listify' ),
				'Company website'                         => __( 'Company website', 'listify' ),
				'Company tagline'                         => __( 'Company tagline', 'listify' ),
				'Brief description about the company'     => __( 'Brief description about the company', 'listify' ),
				'Company Twitter'                         => __( 'Company Twitter', 'listify' ),
				'Company logo'                            => __( 'Company logo', 'listify' ),
				'URL to the company logo'                 => __( 'URL to the company logo', 'listify' ),
				'Company video'                           => __( 'Company video', 'listify' ),

				'WP Job Manager Add-ons'                  => __( 'WP Job Manager Add-ons', 'listify' ),
				'Settings'                                => __( 'Settings', 'listify' ),
				'Add-ons'                                 => __( 'Add-ons', 'listify' ),
				'Approve %s'                              => __( 'Approve %s', 'listify' ),
				'Expire %s'                               => __( 'Expire %s', 'listify' ),
				'%s approved'                             => __( '%s approved', 'listify' ),
				'%s expired'                              => __( '%s expired', 'listify' ),
				'Select category'                         => __( 'Select category', 'listify' ),
				'Position'                                => __( 'Title', 'listify' ),
				'%s updated. View'                        => __( '%s updated. View', 'listify' ),
				'Custom field updated.'                   => __( 'Custom field updated.', 'listify' ),
				'Custom field deleted.'                   => __( 'Custom field deleted.', 'listify' ),
				'%s updated.'                             => __( '%s updated.', 'listify' ),
				'%s restored to revision from %s'         => __( '%1$s restored to revision from %2$s', 'listify' ),
				'%s published. View'                      => __( '%s published. View', 'listify' ),
				'%s saved.'                               => __( '%s saved.', 'listify' ),
				'%s submitted. Preview'                   => __( '%s submitted. Preview', 'listify' ),
				'M j, Y @ G:i'                            => __( 'M j, Y @ G:i', 'listify' ),
				'%s draft updated. Preview'               => __( '%s draft updated. Preview', 'listify' ),
				'Type'                                    => __( 'Type', 'listify' ),
				'Posted'                                  => __( 'Posted', 'listify' ),
				'Expires'                                 => __( 'Expires', 'listify' ),
				'Categories'                              => __( 'Categories', 'listify' ),
				'Featured?'                               => __( 'Featured?', 'listify' ),
				'Filled?'                                 => __( 'Filled?', 'listify' ),
				'Status'                                  => __( 'Status', 'listify' ),
				'Actions'                                 => __( 'Actions', 'listify' ),
				'ID: %d'                                  => __( 'ID: %d', 'listify' ),
				'M j, Y'                                  => __( 'M j, Y', 'listify' ),
				'by a guest'                              => __( 'by a guest', 'listify' ),
				'by %s'                                   => __( 'by %s', 'listify' ),
				'Approve'                                 => __( 'Approve', 'listify' ),
				'View'                                    => __( 'View', 'listify' ),
				'Edit'                                    => __( 'Edit', 'listify' ),
				'Delete'                                  => __( 'Delete', 'listify' ),
				'Listings Per Page'                       => __( 'Listings Per Page', 'listify' ),
				'How many listings should be shown per page by default?' => __( 'How many listings should be shown per page by default?', 'listify' ),
				'Filled Positions'                        => __( 'Filled Positions', 'listify' ),
				'Hide filled positions'                   => __( 'Hide filled positions', 'listify' ),
				'If enabled, filled positions will be hidden.' => __( 'If enabled, filled positions will be hidden.', 'listify' ),
				'Enable categories for listings'          => __( 'Enable categories for listings', 'listify' ),
				'Multi-select Categories'                 => __( 'Multi-select Categories', 'listify' ),
				'Enable category multiselect by default'  => __( 'Enable category multiselect by default', 'listify' ),
				'Category Filter Type'                    => __( 'Category Filter Type', 'listify' ),
				'Account Required'                        => __( 'Account Required', 'listify' ),
				'Submitting listings requires an account' => __( 'Submitting listings requires an account', 'listify' ),
				'Account Creation'                        => __( 'Account Creation', 'listify' ),
				'Allow account creation'                  => __( 'Allow account creation', 'listify' ),
				'Account Role'                            => __( 'Account Role', 'listify' ),
				'Approval Required'                       => __( 'Approval Required', 'listify' ),
				'New submissions require admin approval'  => __( 'New submissions require admin approval', 'listify' ),
				'If enabled, new submissions will be inactive, pending admin approval.' => __( 'If enabled, new submissions will be inactive, pending admin approval.', 'listify' ),
				'Allow Pending Edits'                     => __( 'Allow Pending Edits', 'listify' ),
				'Users can continue to edit pending listings until they are approved by an admin.' => __( 'Users can continue to edit pending listings until they are approved by an admin.', 'listify' ),
				'Submissions awaiting approval can be edited' => __( 'Submissions awaiting approval can be edited', 'listify' ),
				'Allow Published Edits' => __( 'Allow Published Edits', 'listify' ),
				'Allow editing of published listings' => __( 'Allow editing of published listings', 'listify' ),
				'Choose whether published job listings can be edited and if edits require admin approval. When moderation is required, the original job listings will be unpublished while edits await admin approval.' => __( 'Choose whether published job listings can be edited and if edits require admin approval. When moderation is required, the original job listings will be unpublished while edits await admin approval.', 'listify' ),
				'Listings will display for the set number of days, then expire. Leave this field blank if you don\'t want listings to have an expiration date.' => __( 'Listings will display for the set number of days, then expire. Leave this field blank if you don\'t want listings to have an expiration date.', 'listify' ),
				'Users cannot edit' => __( 'Users cannot edit', 'listify' ),
				'Users can edit without admin approval' => __( 'Users can edit without admin approval', 'listify' ),
				'Users can edit, but edits require admin approval' => __( 'Users can edit, but edits require admin approval', 'listify' ),
				'Listing Duration'                        => __( 'Listing Duration', 'listify' ),
				'Application Method'                      => __( 'Contact Method', 'listify' ),
				'Choose the contact method for listings.' => __( 'Choose the contact method for listings.', 'listify' ),
				'Email address or website URL'            => __( 'Email address or website URL', 'listify' ),
				'Email addresses only'                    => __( 'Email addresses only', 'listify' ),
				'Website URLs only'                       => __( 'Website URLs only', 'listify' ),
				'Pages'                                   => __( 'Pages', 'listify' ),
				'Settings successfully saved'             => __( 'Settings successfully saved', 'listify' ),
				'--no page--'                             => __( '--no page--', 'listify' ),
				'Select a page…'                          => __( 'Select a page&hellip;', 'listify' ),
				'Save Changes'                            => __( 'Save Changes', 'listify' ),
				'Setup'                                   => __( 'Setup', 'listify' ),
				'Skip this step'                          => __( 'Skip this step', 'listify' ),
				'All Done!'                               => __( 'All Done!', 'listify' ),
				'Location'                                => __( 'Location', 'listify' ),
				'Employment Type' => __( 'Employment Type', 'listify' ),
				'e.g. "London"'                           => __( 'e.g. "London"', 'listify' ),
				'Leave this blank if the location is not important' => __( 'Leave this blank if the location is not important', 'listify' ),
				'URL or email which applicants use to apply' => __( 'URL or email which applicants use for contact', 'listify' ),
				'URL to the company video'                => __( 'URL to the company video', 'listify' ),
				'Position filled?'                        => __( 'Position filled?', 'listify' ),
				'Feature this listing?'                   => __( 'Feature this listing?', 'listify' ),
				'yyyy-mm-dd'                              => __( 'yyyy-mm-dd', 'listify' ),
				'Posted by'                               => __( 'Posted by', 'listify' ),
				'%s Data'                                 => __( '%s Data', 'listify' ),
				'Use file'                                => __( 'Use file', 'listify' ),
				'Upload'                                  => __( 'Upload', 'listify' ),
				'Add file'                                => __( 'Add file', 'listify' ),
				'Guest user'                              => __( 'Guest user', 'listify' ),
				'Showing %s'                              => __( 'Showing %s', 'listify' ),
				'Showing all %s'                          => __( 'Showing all %s', 'listify' ),
				'located in &ldquo;%s&rdquo;'             => __( 'located in &ldquo;%s&rdquo;', 'listify' ),
				'Search completed. Found %d matching record.' => array(
					__( 'Search completed. Found %d matching record.', 'listify' ),
					__( 'Search completed. Found %d matching records.', 'listify' ),
				),
				'No results found'                        => __( 'No results found', 'listify' ),
				'Query limit reached'                     => __( 'Query limit reached', 'listify' ),
				'Geocoding error'                         => __( 'Geocoding error', 'listify' ),
				'Employer'                                => __( 'Employer', 'listify' ),
				'Search %s'                               => __( 'Search %s', 'listify' ),
				'All %s'                                  => __( 'All %s', 'listify' ),
				'Parent %s'                               => __( 'Parent %s', 'listify' ),
				'Parent %s:'                              => __( 'Parent %s:', 'listify' ),
				'Edit %s'                                 => __( 'Edit %s', 'listify' ),
				'Update %s'                               => __( 'Update %s', 'listify' ),
				'Add New %s'                              => __( 'Add New %s', 'listify' ),
				'New %s Name'                             => __( 'New %s Name', 'listify' ),
				'Add New'                                 => __( 'Add New', 'listify' ),
				'Add %s'                                  => __( 'Add %s', 'listify' ),
				'New %s'                                  => __( 'New %s', 'listify' ),
				'View %s'                                 => __( 'View %s', 'listify' ),
				'No %s found'                             => __( 'No %s found', 'listify' ),
				'No %s found in trash'                    => __( 'No %s found in trash', 'listify' ),
				'This is where you can create and manage %s.' => __( 'This is where you can create and manage %s.', 'listify' ),
				'Expired'                                 => array(
					__( 'Expired', 'listify' ),
					__( 'Expired (%s)', 'listify' ),
				),
				'Invalid ID'                              => __( 'Invalid ID', 'listify' ),
				'This position has already been filled'   => __( 'This position has already been filled', 'listify' ),
				'%s has been filled'                      => __( '%s has been filled', 'listify' ),
				'This position is not filled'             => __( 'This position is not filled', 'listify' ),
				'%s has been marked as not filled'        => __( '%s has been marked as not filled', 'listify' ),
				'%s has been deleted'                     => __( '%s has been deleted', 'listify' ),
				'Job Title'                               => sprintf( __( '%s Name', 'listify' ), $this->label( 'singular' ) ),
				'Date Posted'                             => __( 'Date Posted', 'listify' ),
				'Date Expires'                            => __( 'Date Expires', 'listify' ),
				'Load more listings'                      => sprintf( __( 'Load More %s', 'listify' ), $this->label( 'plural' ) ),
				'Recent %s'                               => __( 'Recent %s', 'listify' ),
				'Keyword'                                 => __( 'Keyword', 'listify' ),
				'Number of listings to show'              => __( 'Number of listings to show', 'listify' ),
				'Invalid listing'                         => __( 'Invalid listing', 'listify' ),
				'Save changes'                            => __( 'Save changes', 'listify' ),
				'Your changes have been saved.'           => __( 'Your changes have been saved.', 'listify' ),
				'View &rarr;'                             => __( 'View &rarr;', 'listify' ),
				'Submit Details'                          => __( 'Submit Details', 'listify' ),
				'Preview'                                 => __( 'Preview', 'listify' ),
				'Done'                                    => __( 'Done', 'listify' ),
				'you@yourdomain.com'                      => __( 'you@yourdomain.com', 'listify' ),
				'http://'                                 => __( 'http://', 'listify' ),
				'Enter an email address or website URL'   => __( 'Enter an email address or website URL', 'listify' ),
				'Description'                             => __( 'Description', 'listify' ),
				'Enter the name of the company'           => __( 'Enter the name of the company', 'listify' ),
				'Website'                                 => __( 'Website', 'listify' ),
				'Tagline'                                 => __( 'Tagline', 'listify' ),
				'Briefly describe your company'           => __( 'Briefly describe your company', 'listify' ),
				'Video'                                   => __( 'Video', 'listify' ),
				'A link to a video about your company'    => __( 'A link to a video about your company', 'listify' ),
				'Twitter username'                        => __( 'Twitter username', 'listify' ),
				'@yourcompany'                            => __( '@yourcompany', 'listify' ),
				'Logo'                                    => __( 'Logo', 'listify' ),
				'%s is a required field'                  => __( '%s is a required field', 'listify' ),
				'%s is invalid'                           => __( '%s is invalid', 'listify' ),
				'Please enter a valid application email address' => __( 'Please enter a valid contact email address', 'listify' ),
				'Please enter a valid application URL'    => __( 'Please enter a valid application URL', 'listify' ),
				'Please enter a valid application email address or URL' => __( 'Please enter a valid contact email address or URL', 'listify' ),
				'You must be signed in to post a new listing.' => __( 'You must be signed in to post a new listing.', 'listify' ),
				'Submit Listing'                          => __( 'Submit Listing', 'listify' ),
				'Edit listing'                            => __( 'Edit listing', 'listify' ),
				'\%s\ (filetype %s) needs to be one of the following file types: %s' => __( '\%1$s\ (filetype %2$s) needs to be one of the following file types: %3$s', 'listify' ),
				'Your account'                            => __( 'Your account', 'listify' ),
				'You are currently signed in as <strong>%s</strong>.' => __( 'You are currently signed in as %s.', 'listify' ),
				'Sign out'                                => __( 'Sign out', 'listify' ),
				'Have an account?'                        => __( 'Have an account?', 'listify' ),
				'Sign in'                                 => __( 'Sign in', 'listify' ),
				'optionally'                              => __( 'optionally', 'listify' ),
				'You must sign in to create a new listing.' => __( 'You must sign in to create a new listing.', 'listify' ),
				'Your email'                              => __( 'Your email', 'listify' ),
				'(optional)'                              => __( '(optional)', 'listify' ),
				'%s ago'                                  => __( '%s ago', 'listify' ),
				'No more results found.'                  => __( 'No more results found.', 'listify' ),
				'Posted %s ago'                           => __( 'Posted %s ago', 'listify' ),
				'This position has been filled'           => __( 'This position has been filled', 'listify' ),
				'This listing has expired'                => __( 'This listing has expired', 'listify' ),
				'remove'                                  => __( 'remove', 'listify' ),
				'or'                                      => __( 'or', 'listify' ),
				'Maximum file size: %s.'                  => __( 'Maximum file size: %s.', 'listify' ),
				'Apply using webmail:'                    => __( 'Apply using webmail:', 'listify' ),
				'To apply for this job please visit the following URL: <a href=\"%1$s\" target=\"_blank\">%1$s &rarr;</a>' => __( 'To contact this listing owner please visit the following URL: <a href=\"%1$s\" target=\"_blank\">%1$s %rarr;</a>', 'listify' ),

				'Apply for job'                           => array(
					__( 'Contact %s', 'listify' ),
					array( $this->label( 'singular' ) ),
				),

				'You need to be signed in to manage your listings.' => __( 'You need to be signed in to manage your listings.', 'listify' ),
				'You do not have any active listings.'    => __( 'You do not have any active listings.', 'listify' ),
				'Mark not filled'                         => __( 'Mark not filled', 'listify' ),
				'Mark filled'                             => __( 'Mark filled', 'listify' ),
				'Relist'                                  => __( 'Relist', 'listify' ),
				'Keywords'                                => __( 'What are you looking for?', 'listify' ),
				'Category'                                => __( 'Category', 'listify' ),
				'Any category'                            => __( 'All categories', 'listify' ),
				'Company Details'                         => __( 'Company Details', 'listify' ),
				'%s submitted successfully. Your listing will be visible once approved.' => __( '%s submitted successfully. Your listing will be visible once approved.', 'listify' ),
				'Draft'                                   => __( 'Draft', 'listify' ),
				'Preview'                                 => __( 'Preview', 'listify' ),
				'Pending approval'                        => __( 'Pending approval', 'listify' ),
				'Pending payment'                         => __( 'Pending payment', 'listify' ),
				'Active'                                  => __( 'Active', 'listify' ),
				'Reset'                                   => __( 'Reset', 'listify' ),
				'RSS'                                     => __( 'RSS', 'listify' ),
				'Your email address isn’t correct.'       => __( 'Your email address isn&#39;t correct.', 'listify' ),
				'This email is already registered, please choose another one.' => __( 'This email is already registered, please choose another one.', 'listify' ),
				'Choose a category&hellip;'               => __( 'Choose a category&hellip;', 'listify' ),
				'Inactive'                                => __( 'Inactive', 'listify' ),
				'Application via \%s\ listing on %s'      => __( 'Application via \%1$s\ listing on %2$s', 'listify' ),
				'Anywhere'                                => __( 'Anywhere', 'listify' ),
				'Are you sure you want to delete this listing?' => __( 'Are you sure you want to delete this listing?', 'listify' ),
				'Your listings are shown in the table below.' => __( 'Your listings are shown in the table below.', 'listify' ),
				'Listing Expires'                         => __( 'Listing Expires', 'listify' ),
				'If you don\'t have an account you can %screate one below by entering your email address/username.' => __( 'If you don\'t have an account you can %screate one below by entering your email address/username.', 'listify' ),
				'Your account details will be confirmed via email.' => __( 'Your account details will be confirmed via email.', 'listify' ),
				'To apply for this job please visit the following URL: <a href="%1$s" target="_blank">%1$s &rarr;</a>' => __( 'To contact this listing owner please visit the following URL: <a href="%1$s" target="_blank">%1$s &rarr;</a>', 'listify' ),
				'To apply for this job <strong>email your details to</strong> <a class="job_application_email" href="mailto:%1$s%2$s">%1$s</a>' => __( 'To contact this listing <strong>email your details to</strong> <a class="job_application_email" href="mailto:%1$s%2$s">%1$s</a>', 'listify' ),
				'You are editing an existing job. %s'     => __( 'You are editing an existing listing. %s', 'listify' ),
				'Create A New Job'                        => __( 'Create a new Listing', 'listify' ),
				'\"%s\" check failed. Please try again.'  => __( '"%s" check failed. Please try again.', 'listify' ),
				'Licenses'                                => __( 'Licenses', 'listify' ),
				'Choose how you want the published date for jobs to be displayed on the front-end.' => __( 'Choose how you want the published date for jobs to be displayed on the front-end.', 'listify' ),
				'Date Format' => __( 'Date Format', 'listify' ),
				'Relative to the current date (e.g., 1 day, 1 week, 1 month ago)' => __( 'Relative to the current date (e.g., 1 day, 1 week, 1 month ago)' ),
				'Default date format as defined in Settings' => __( 'Default date format as defined in Settings', 'listify' ),
				'Google Maps API Key' => __( 'Google Maps API Key', 'listify' ),
				'Google requires an API key to retrieve location information for job listings. Acquire an API key from the <a href=\"%s\">Google Maps API developer site</a>.' => 'Google requires an API key to retrieve location information for listings. Acquire an API key from the <a href=\"%s\">Google Maps API developer site</a>.',
				'Delete Data On Uninstall' => __( 'Delete Data On Uninstall', 'listify' ),
				'Delete WP Job Manager data when the plugin is deleted. Once removed, this data cannot be restored.' => __( 'Delete WP Job Manager data when the plugin is deleted. Once removed, this data cannot be restored.', 'listify' ),
				'Includes account creation on the listing submission form, to allow non-registered users to create an account and submit a job listing simultaneously.' => __( 'Includes account creation on the listing submission form, to allow non-registered users to create an account and submit a listing simultaneously.', 'listify' ),
				'Automatically generates usernames for new accounts from the registrant\'s email address. If this is not enabled, a \"username\" field will display instead.' => __( 'Automatically generates usernames for new accounts from the registrant\'s email address. If this is not enabled, a \"username\" field will display instead.', 'listify' ),
				'Email new users a link to set a password' => __( 'Email new users a link to set a password', 'listify' ),
				'Sends an email to the user with their username and a link to set their password. If this is not enabled, a \"password\" field will display instead, and their email address won\'t be verified.' => __( 'Sends an email to the user with their username and a link to set their password. If this is not enabled, a \"password\" field will display instead, and their email address won\'t be verified.', 'listify' ),
				'Any new accounts created during submission will have this role. If you haven\'t enabled account creation during submission in the options above, your own method of assigning roles will apply.' => __( 'Any new accounts created during submission will have this role. If you haven\'t enabled account creation during submission in the options above, your own method of assigning roles will apply.', 'listify' ),
				'Moderate New Listings' => __( 'Moderate New Listings', 'listify' ),
				'Require admin approval of all new listing submissions' => __( 'Require admin approval of all new listing submissions', 'listify' ),
				'Sets all new submissions to "pending." They will not appear on your site until an admin approves them.' => __( 'Sets all new submissions to "pending." They will not appear on your site until an admin approves them.', 'listify' ),
				'Send a notice to the site administrator when a new job is submitted on the frontend.' => __( 'Send a notice to the site administrator when a new listing is submitted on the frontend.', 'listify' ),
				'Send a notice to the site administrator when a job is updated on the frontend.' => __( 'Send a notice to the site administrator when a listing is updated on the frontend.', 'listify' ),
				'Admin Notice of Expiring Job Listings' => __( 'Admin Notice of Expiring Listings', 'listify' ),
				'Send notices to the site administrator before a job listing expires.' => __( 'Send notices to the site administrator before a listing expires.', 'listify' ),
				'Employer Notice of Expiring Job Listings' => __( 'Employer Notice of Expiring Listings', 'listify' ),
				'Send notices to employers before a job listing expires.' => __( 'Send notices to employers before a listing expires.', 'listify' ),
				'The following job listing is expiring today from <a href="%s">%s</a>.' => __( 'The following listing is expiring today from <a href="%s">%s</a>.', 'listify' ),
				'The following job listing is expiring soon from <a href="%s">%s</a>.' => __( 'The following listing is expiring soon from <a href="%s">%s</a>.', 'listify' ),
				'Visit <a href="%s">WordPress admin</a> to manage the listing.' => __( 'Visit <a href="%s">WordPress admin</a> to manage the listing.', 'listify' ),
				'It has been published and is now available to the public.' => __( 'It has been published and is now available to the public.', 'listify' ),
				'It is awaiting approval by an administrator in <a href="%s">WordPress admin</a>.' => __( 'It is awaiting approval by an administrator in <a href="%s">WordPress admin</a>.', 'listify' ),
				'A new listing has been submitted to <a href="%s">%s</a>.' => __( 'A new job listing has been submitted to <a href="%s">%s</a>.', 'listify' ),
				'A job listing has been updated on <a href="%s">%s</a>.' => __( 'A listing has been updated on <a href="%s">%s</a>.', 'listify' ),
				'The job listing is not publicly available until the changes are approved by an administrator in the site\'s <a href="%s">WordPress admin</a>.' => __( 'The listing is not publicly available until the changes are approved by an administrator in the site\'s <a href="%s">WordPress admin</a>.', 'listify' ),
				'The changes have been published and are now available to the public.' => __( 'The changes have been published and are now available to the public.', 'listify' ),
				'The following job listing is expiring today from <a href="%s">%s</a>.' => __( 'The following listing is expiring today from <a href="%s">%s</a>.', 'listify' ),
				'The following job listing is expiring soon from <a href="%s">%s</a>.' => __( 'The following job listing is expiring soon from <a href="%s">%s</a>.', 'listify' ),
				'Visit the <a href="%s">job listing dashboard</a> to manage the listing.' => __( 'Visit the <a href="%s">listing dashboard</a> to manage the listing.', 'listify' ),
				'Hide expired listings in job archives/search' => __( 'Hide expired listings in archives/search', 'listify' ),
				'Expired job listings will not be searchable.' => __( 'Expired listings will not be searchable.', 'listify' ),
				'This lets users select from a list of categories when submitting a job. Note: an admin has to create categories before site users can select them.' => __( 'This lets users select from a list of categories when submitting a listing. Note: an admin has to create categories before site users can select them.', 'listify' ),
				'The category selection box will default to allowing multiple selections on the [jobs] shortcode. Without this, users will only be able to select a single category when submitting jobs.' => __( 'The category selection box will default to allowing multiple selections on the [jobs] shortcode. Without this, users will only be able to select a single category when submitting listings.', 'listify' ),
				'Determines the logic used to display jobs when selecting multiple categories.' => __( 'Determines the logic used to display listings when selecting multiple categories.', 'listify' ),
				'This lets users select from a list of types when submitting a job. Note: an admin has to create types before site users can select them.' => __( 'This lets users select from a list of types when submitting a listing. Note: an admin has to create types before site users can select them.', 'listify' ),
				'Choose how you want the published date for jobs to be displayed on the front-end.' => __( 'Choose how you want the published date for listings to be displayed on the front-end.', 'listify' ),
				'Limits job listing submissions to registered, logged-in users.' => __( 'Limits listing submissions to registered, logged-in users.', 'listify' ),
				'New Job Listing Submitted: %s' => __( 'New Listing Submitted: %s', 'listify' ),
				'Job Listing Updated: %s' => __( 'Listing Updated: %s', 'listify' ),
				'Job Listing Expiring: %s' => __( 'Listing Expiring: %s', 'listify' ),
			),
			'wp-job-manager-listing-labels'       => array(
				'job-tag' => array(
					__( '%s-tag', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),
			),
			'wp-job-manager-locations'            => array(
				'Job Regions'                    => array(
					__( '%s Regions', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Job Region'                     => array(
					__( '%s Region', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'job-region'                     => array(
					__( '%s-region', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),
				'Display a list of job regions.' => array(
					__( 'Display a list of %s regions.', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),
			),
			'wp-job-manager-wc-paid-listings'     => array(
				'Choose a package before entering job details' => sprintf( __( 'Choose a package before entering %s details', 'listify' ), $this->label( 'singular' ) ),
				'Choose a package after entering job details' => sprintf( __( 'Choose a package after entering %s details', 'listify' ), $this->label( 'singular' ) ),
				'Choose a package'         => __( 'Choose a package', 'listify' ),
				'Purchase Package:'        => __( 'Purchase Package:', 'listify' ),
				'Listing Details &rarr;'   => __( 'Listing Details &rarr;', 'listify' ),
				'%s job posted out of %d'  => array(
					__( '%1$s listing posted out of %2$d', 'listify' ),
					__( '%1$s listings posted out of %2$d', 'listify' ),
				),
				'%s job posted'            => array(
					__( '%s listing posted', 'listify' ),
					__( '%s listings posted', 'listify' ),
				),
				'%s for %s job'            => array(
					__( '%1$s for %2$s listing', 'listify' ),
					__( '%1$s for %2$s listings', 'listify' ),
				),
				'Job Package'              => array(
					__( '%s Package', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Job Package Subscription' => array(
					__( '%s Package Subscription', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Job Listing'              => array(
					__( '%s', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Job listing limit'        => array(
					__( '%s limit', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Job listing duration'     => array(
					__( '%s duration', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'The number of days that the job listing will be active.' => array(
					__( 'The number of days that the %s will be active', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),
				'Feature job listings?'    => array(
					__( 'Feature %s?', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),
				'Feature this job listing - it will be styled differently and sticky.' => array(
					__( 'Feature this %s -- it will be styled differently and sticky.', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),
				'My Job Packages'          => array(
					__( 'My %s Packages', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Jobs Remaining'           => array(
					__( '%s Remaining', 'listify' ),
					array( $this->label( 'plural' ) ),
				),
			),
			'wp-job-manager-simple-paid-listings' => array(
				'Job #%d Payment Update' => __( '#%d Payment Update', 'listify' ),
			),
			'wp-job-manager-alerts'               => array(
				'Jobs matching your "%s" alert:' => __( 'Results for your "%s" alert:', 'listify' ),
				'Job Alert Results Matching "%s' => __( 'Results Matching "%s', 'listify' ),
				'No jobs were found matching your search. Login to your account to change your alert criteria' => __( 'No results were found matching your search. Login to your account to change your alert criteria', 'listify' ),
				'This job alert will automatically stop sending after %s.' => __( 'This alert will automatically stop sending after %s.', 'listify' ),
				'No jobs found'                  => array(
					__( 'No %s found', 'listify' ),
					array( $this->label( 'plural', true ) ),
				),
				'Optionally add a keyword to match jobs against' => array(
					__( 'Optionally add a keyword to match %s against', 'listify' ),
					array( $this->label( 'plural', true ) ),
				),
				'Job Type'                       => array(
					__( '%s Type', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Job Region'                     => array(
					__( '%s Region', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Any job type'                   => array(
					__( 'Any %s type', 'listify' ),
					array( $this->label( 'singular', true ) ),
				),
				'Job Type:'                      => array(
					__( '%s Type:', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
				'Your job alerts are shown in the table below. Your alerts will be sent to %s.' => __( 'Your alerts are shown in the table below. The alerts will be sent to %s.', 'listify' ),
				'Alert me to jobs like this'     => sprintf( __( 'Alert me of %s like this', 'listify' ), $this->label( 'plural', true ) ),
			),
			'wp-job-manager-extended-location'    => array(
				'Job Location' => array(
					__( '%s Location', 'listify' ),
					array( $this->label( 'singular' ) ),
				),
			),
		);

		$this->strings = apply_filters( 'listify_strings', $strings );

		return $this->strings;
	}

}

$GLOBALS['listify_strings'] = new Listify_Strings();
